package com.anf.ws.ar.xml.utils;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.URIResolver;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.SAXException;

public class XmlUtils {
	
	private XmlUtils() {}
	
	public static Document readDOMDocument(InputStream input)
			throws ParserConfigurationException, FactoryConfigurationError, SAXException, IOException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		// to be compliant, completely disable DOCTYPE declaration:
		factory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
		// or completely disable external entities declarations:
		factory.setFeature("http://xml.org/sax/features/external-general-entities", false);
		factory.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
		// or prohibit the use of all protocols by external entities:
		factory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
		factory.setAttribute(XMLConstants.ACCESS_EXTERNAL_SCHEMA, "");
		// or disable entity expansion but keep in mind that this doesn't prevent fetching external entities
		// and this solution is not correct for OpenJDK < 13 due to a bug: https://bugs.openjdk.java.net/browse/JDK-8206132
		factory.setExpandEntityReferences(false);
		
		DocumentBuilder docBuilder = factory.newDocumentBuilder();
		return docBuilder.parse(input);
	}

	public static void transform(InputStream xmlInput, InputStream xslInput, OutputStream outputData)
			throws TransformerException, TransformerFactoryConfigurationError {
		transform(xmlInput, xslInput, outputData, (URIResolver) null);
	}

	public static void transform(InputStream xmlInput, InputStream xslInput, OutputStream outputData,
			URIResolver uriResolver) throws TransformerException, TransformerFactoryConfigurationError {
		transform(xmlInput, xslInput, outputData, uriResolver, (Map<String,String>) null);
	}

	public static void transform(InputStream xmlInput, InputStream xslInput, OutputStream outputData,
			URIResolver uriResolver, Map<String,String> parameters) throws TransformerException, TransformerFactoryConfigurationError {
		StreamSource xmlData = new StreamSource(xmlInput);
		StreamSource xslData = new StreamSource(xslInput);
		StreamResult resultData = new StreamResult(outputData);
		transform(xmlData, xslData, resultData, uriResolver, parameters);
	}

	public static void transform(Reader xmlInput, Reader xslInput, Writer outputData)
			throws TransformerException, TransformerFactoryConfigurationError {
		transform(xmlInput, xslInput, outputData, (URIResolver) null);
	}

	public static void transform(Reader xmlInput, Reader xslInput, Writer outputData, URIResolver uriResolver)
			throws TransformerException, TransformerFactoryConfigurationError {
		transform(xmlInput, xslInput, outputData, uriResolver);
	}

	public static void transform(Reader xmlInput, Reader xslInput, Writer outputData, URIResolver uriResolver,
			Map<String,String> parameters) throws TransformerException, TransformerFactoryConfigurationError {
		StreamSource xmlData = new StreamSource(xmlInput);
		StreamSource xslData = new StreamSource(xslInput);
		StreamResult resultData = new StreamResult(outputData);
		transform(xmlData, xslData, resultData, uriResolver, parameters);
	}

	public static void transform(Node xmlNode, Node xslNode, Node outputNode)
			throws TransformerException, TransformerFactoryConfigurationError {
		transform(xmlNode, xslNode, outputNode, (URIResolver) null);
	}

	public static void transform(Node xmlNode, Node xslNode, Node outputNode, URIResolver uriResolver)
			throws TransformerException, TransformerFactoryConfigurationError {
		transform(xmlNode, xslNode, outputNode, uriResolver, (Map<String,String>) null);
	}

	public static void transform(Node xmlNode, Node xslNode, Node outputNode, URIResolver uriResolver, Map<String,String> parameters)
			throws TransformerException, TransformerFactoryConfigurationError {
		DOMSource xmlData = new DOMSource(xmlNode);
		DOMSource xslData = new DOMSource(xslNode);
		DOMResult resultData = new DOMResult(outputNode);
		transform(xmlData, xslData, resultData, uriResolver, parameters);
	}

	public static void transform(Source xmlData, Source xslData, Result resultData)
			throws TransformerException, TransformerFactoryConfigurationError {
		transform(xmlData, xslData, resultData, (URIResolver) null);
	}

	public static void transform(Source xmlData, Source xslData, Result resultData, URIResolver uriResolver)
			throws TransformerException, TransformerFactoryConfigurationError {
		transform(xmlData, xslData, resultData, uriResolver, (Map<String,String>) null);
	}

	public static void transform(Source xmlData, Source xslData, Result resultData, URIResolver uriResolver,
			Map<String,String> parameters) throws TransformerException, TransformerFactoryConfigurationError {
		TransformerFactory tfactory = TransformerFactory.newInstance();
		// to be compliant, prohibit the use of all protocols by external entities:
		tfactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
		tfactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_STYLESHEET, "");
		if (uriResolver != null) {
			tfactory.setURIResolver(uriResolver);
		}

		Transformer transformer = tfactory.newTransformer(xslData);
		if (parameters != null) {
			Iterator<String> keysIt = parameters.keySet().iterator();

			while (keysIt.hasNext()) {
				String key = keysIt.next();
				transformer.setParameter(key, parameters.get(key));
			}
		}

		transformer.transform(xmlData, resultData);
	}

	public static void serializeNode(Node node, Writer output)
			throws TransformerException, TransformerFactoryConfigurationError {
		serializeNode(node, output, "ISO-8859-1");
	}

	public static void serializeNode(Node node, Writer output, String encoding)
			throws TransformerException, TransformerFactoryConfigurationError {
		TransformerFactory tfactory = TransformerFactory.newInstance();
		// to be compliant, prohibit the use of all protocols by external entities:
		tfactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
		tfactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_STYLESHEET, "");
		Transformer transformer = tfactory.newTransformer();
		DOMSource source = new DOMSource(node);
		StreamResult result = new StreamResult(output);
		transformer.setOutputProperty("encoding", encoding);
		transformer.transform(source, result);
	}

	public static void serializeNode(Node node, OutputStream output, String encoding)
			throws TransformerException, TransformerFactoryConfigurationError {
		OutputStreamWriter writer = new OutputStreamWriter(output);
		serializeNode(node, writer, encoding);
	}

	public static void serializeNode(Node node, OutputStream output)
			throws TransformerException, TransformerFactoryConfigurationError {
		serializeNode(node, output, "ISO-8859-1");
	}

	public static Element getFirstChildElement(Element elem) {
		Node node=elem.getFirstChild();
		while(node != null && !(node instanceof Element))
			node = node.getNextSibling();
		return (Element) node;
	}

	public static Element getFirstSiblingElement(Element elem) {
		return getFirstChildElement(elem);
	}

	public static void replaceContent(Element elem, String value) {
		if (elem.getFirstChild() != null) {
			elem.getFirstChild().setNodeValue(value);
		} else {
			Text txtNode = elem.getOwnerDocument().createTextNode(value);
			elem.appendChild(txtNode);
		}

	}

	public static Node getChildNodeWithName(Node parent, String childName) {
		NodeList children = parent.getChildNodes();
		Node soughtNode = null;
		
		boolean found = false;
		for (int i = 0; !found && i < children.getLength(); i++) {
			Node currentChild = children.item(i);
			String currentName = currentChild.getNodeName();
			found = childName.equals(currentName);
			if (found) {
				soughtNode = currentChild;
			}
		}

		return soughtNode;
	}

	public static boolean equals(Node xml1, Node xml2) {
		if(xml1.getNodeType() != xml2.getNodeType())
			return false;
		if(!Objects.equals(xml1.getNodeName(), xml2.getNodeName()))
			return false;
		
		if(!Objects.equals(xml1.getNodeValue(), xml2.getNodeValue()))
			return false;
		if(!compareNodeValues(xml1, xml2))
			return false;
		return compareChilds(xml1, xml2);
	}

	private static boolean compareNodeValues(Node xml1, Node xml2) {
		NamedNodeMap attrs1 = xml1.getAttributes();
		NamedNodeMap attrs2 = xml2.getAttributes();
		if (attrs1 == null ^ attrs2 == null) {
			return false;
		} 

		if (attrs1 != null && attrs1.getLength() != attrs2.getLength()) {
			return false;
		}

		if (attrs1 != null) {

			for (int i = 0; i < attrs1.getLength(); ++i) {
				Node attrFrom1 = attrs1.item(i);
				String attrName = attrFrom1.getNodeName();
				Node attrFrom2 = attrs2.getNamedItem(attrName);
				if (attrFrom2 == null) {
					return false;
				}

				if (!attrFrom1.getNodeValue().equals(attrFrom2.getNodeValue())) {
					return false;
				}
			}
		}

		return true;
	}

	private static boolean compareChilds(Node xml1, Node xml2) {
		NodeList children1 = xml1.getChildNodes();
		NodeList children2 = xml2.getChildNodes();
		if (children1.getLength() != children2.getLength()) {
			return false;
		} else {
			for (int i = 0; i < children1.getLength(); ++i) {
				Node childFrom1 = children1.item(i);
				Node childFrom2 = children2.item(i);
				if (!equals(childFrom1, childFrom2)) {
					return false;
				}
			}

			return true;
		}
	}
}