package com.anf.ws.ar.xml.attributes.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.x500.style.BCStyle;

import com.anf.ra.xml.ApplicationForm;
import com.anf.ra.xml.ApplicationForm.Applicant.Attributes.Attribute;
import com.anf.ws.ar.xml.attributes.AttributeValidator;
import com.anf.ws.ar.xml.attributes.AttributesProfile;
import com.anf.ws.ar.xml.service.params.XmlBuildParams;
import com.anf.ws.ar.xml.types.CertificateType;

public class SignAttributesProfile implements AttributesProfile {

	private static final Logger log = LogManager.getLogger(SignAttributesProfile.class);

	protected XmlBuildParams params;
	protected CertificateType certType;

	String country=null;

	final List<String> typeReferences = List.of("VAT","NTR","PSD","LEI");

	public SignAttributesProfile(XmlBuildParams params, CertificateType certType) {
		super();
		this.params = params;
		this.certType = certType;
	}

	public List<Attribute> build() {
		List<Attribute> attributes = new ArrayList<>();
		List<ApplicationForm.Applicant.Attributes.Attribute> xmlAttributes = params.getXmlInfo().obtenerAtributos();

		if (xmlAttributes != null && !xmlAttributes.isEmpty()) {

			detectCountryField(xmlAttributes);

			fillAttributes(xmlAttributes, attributes);			

			updateSerialNumber(attributes);

			AttributeValidator.validateAttributes(attributes);

		}
		return attributes;
	}

	private void fillAttributes(
			List<ApplicationForm.Applicant.Attributes.Attribute> xmlAttributes,
			List<Attribute> attributes) {

		Set<String> oidsAdded=new TreeSet<>();

		xmlAttributes.stream()
		.filter(a->params.isTest() || AttributeValidator.validAttribute(certType, country, a) || printNotValidAttribute(a))
		.forEach(a->{
			ApplicationForm.Applicant.Attributes.Attribute attribute = attribute(a.getAttributeType().trim(),a.getAttributeValue().trim());

			String type = attribute.getAttributeType();
			if(type.equals(BCStyle.GIVENNAME.getId()) || type.equals(BCStyle.SURNAME.getId()) || type.equals(BCStyle.SERIALNUMBER.getId()))
				attribute.setAttributeValue(attribute.getAttributeValue().toUpperCase());

			attributes.add(attribute);
			oidsAdded.add(attribute.getAttributeType());
		});		

		if(oidsAdded.add(BCStyle.CN.getId())) {
			String commonName = commonName();
			if(commonName!=null && !commonName.isBlank())
				attributes.add(attribute(BCStyle.CN.getId(),commonName));
		}

		String organizationalUnit = organizationalUnit();
		if(organizationalUnit!=null && !organizationalUnit.isBlank())
			attributes.add(attribute(BCStyle.OU.getId(),organizationalUnit));
	}

	private boolean printNotValidAttribute(Attribute a) {
		log.info("not valid attribute: oid: {} value: {} for cert type: {} and country: {}", a.getAttributeType(), a.getAttributeValue(), certType, country);
		return false;
	}


	private void detectCountryField(List<ApplicationForm.Applicant.Attributes.Attribute> xmlAttributes) {
		xmlAttributes.stream().filter(a->a.getAttributeType().equals(BCStyle.C.getId())).findFirst().ifPresent(a->{
			if(a.getAttributeValue()!=null && a.getAttributeValue().toLowerCase().startsWith("esp")){
				a.setAttributeValue("ES");
			}
			if(a.getAttributeValue()!=null && !a.getAttributeValue().isBlank()){
				country=a.getAttributeValue().trim().toUpperCase();
			}
		});
	}

	protected void updateSerialNumber(List<Attribute> attributes) {
		if(country==null)
			return;
		if(!country.equals("EC") && !country.equals("PE") || ecuadorProfiles()) {

			if(certType.includeOidDNI()) {
				attributes.add(attribute("1.3.6.1.4.1.18838.1.1",getAttribute(BCStyle.SERIALNUMBER).trim().toUpperCase()));
			}

			for (ApplicationForm.Applicant.Attributes.Attribute a : attributes) {
				if (a.getAttributeType().equals(BCStyle.SERIALNUMBER.getId()) && 
						!a.getAttributeValue().startsWith("TIN") && 
						!a.getAttributeValue().startsWith("IDC") &&
						certType!=CertificateType.SELLO_ELECTRONICO_EC) {
					a.setAttributeValue("IDC"+country+"-"+a.getAttributeValue().trim());
				}
				if (a.getAttributeType().equals(BCStyle.ORGANIZATION_IDENTIFIER.getId())) {
					a.setAttributeValue(fixCIF(a.getAttributeValue().trim(),country));
				}
			}
		}else if(country.equals("EC") && !ecuadorProfiles()) {
			attributes.add(attribute("1.3.6.1.4.1.37442.10.4",getAttribute(BCStyle.SERIALNUMBER).trim()));
		}
	}


	private boolean ecuadorProfiles() {
		return certType.match(
				CertificateType.PERSONA_NATURAL_EC,
				CertificateType.REP_LEGAL_EC,
				CertificateType.MIEMBRO_EMPRESA_EC,
				CertificateType.SELLO_ELECTRONICO_EC);
	}

	private String fixCIF(String strValue, String country) {
		String cif=strValue;
		boolean ecuadorProfiles = ecuadorProfiles();
		boolean includeTypeIdentifier = country!=null && (!country.equals("EC") && !country.equals("PE") || ecuadorProfiles);

		String typeId="VAT";
		if(certType==CertificateType.PERSONA_NATURAL_EC)
			typeId="TIN";

		int separator = cif.indexOf("-");
		if(separator==-1) {
			if(includeTypeIdentifier)
				return typeId+country+"-"+cif;
			return cif;
		}

		String secondPart=cif.substring(separator+1);
		if(includeTypeIdentifier) {
			String part=cif.substring(0,separator);
			if(typeReferences.stream().anyMatch(type->(type+country).equals(part)))
				return cif;
			return typeId+country+"-"+secondPart;
		}
		return secondPart;
	}



	private String getAttribute(ASN1ObjectIdentifier oid) {
		return params.getXmlInfo().getAttributeValue(oid.getId());
	}


	private ApplicationForm.Applicant.Attributes.Attribute attribute(String key, String value){
		return new ApplicationForm.Applicant.Attributes.Attribute(key, value);
	}

	public String getCountry() {
		return country;
	}


	protected String commonName() {
		String commonName = "";
		String dni = getAttribute(BCStyle.SERIALNUMBER);
		if(dni!=null)
			dni=dni.trim().toUpperCase();
		String nombre = getAttribute(BCStyle.GIVENNAME);
		if(nombre!=null)
			nombre=nombre.trim().toUpperCase();
		String apellidos = getAttribute(BCStyle.SURNAME);
		if(apellidos!=null)
			apellidos=apellidos.trim().toUpperCase();

		String colegiadoOu = getAttribute(BCStyle.OU);
		String numColegiado=colegiadoOu.replace("Colegiado ", "").trim();

		switch (certType) {
		case 
		PERSONA_NATURAL_EC,
		PERSONA_FISICA,
		PERSONA_FISICA_CORP,
		OPERADOR_AR,
		OPERADOR_AR_2,
		RDE,
		OPERADOR_PKI,
		OPERADOR_PKI_ADMIN,
		EMPLEADO_PUBLICO_ALTO,
		EMPLEADO_PUBLICO_MEDIO:
			if (dni!=null && !dni.isBlank()) {
				commonName = nombre + " " + apellidos;
				if(!country.equals("EC") && !country.equals("PE"))
					commonName+=" - " +dni;
			}
		break;
		case 
		CORPORATIVO_COLEGIADO:
			if (dni!=null && !dni.isBlank()) {
				commonName = nombre + " " + apellidos;
				if(!country.equals("EC") && !country.equals("PE"))
					commonName+=" / num" +numColegiado;
			}
		break;
		case 
		REP_LEGAL_EC,
		REP_LEGAL_PJ,
		REP_LEGAL_ADMIN_UNICOS,
		REP_LEGAL_SIN_PJ:
			if (dni!=null && !dni.isBlank()) {
				String cif = "";
				if (!country.equals("EC") && !country.equals("PE")) {
					cif = " " + "(R: " + getAttribute(BCStyle.ORGANIZATION_IDENTIFIER) + ")";
				}
				commonName = dni + " " + nombre + " " + apellidos + cif;
			}
		break;
		default:
			commonName = dni+" "+nombre + " " + apellidos;
			break;
		}
		return commonName.trim().toUpperCase();
	}

	protected String organizationalUnit() {
		String oun = "";

		if(params.getOuProfile()!=null) {
			oun=params.getOuProfile();
		}else if (null != certType) {
			oun=certType.toString();
			if (oun!=null && country!=null && country.equalsIgnoreCase("EC") && 
					certType.includeCountry()) {
				oun += " " + country;
			}
		}
		return oun;
	}
}
