/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.asic.common;

import eu.europa.esig.dss.asic.common.ASiCContent;
import eu.europa.esig.dss.asic.common.ASiCParameters;
import eu.europa.esig.dss.asic.common.ContainerEntryDocument;
import eu.europa.esig.dss.asic.common.ZipUtils;
import eu.europa.esig.dss.enumerations.ASiCContainerType;
import eu.europa.esig.dss.enumerations.MimeType;
import eu.europa.esig.dss.enumerations.MimeTypeEnum;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.DigestDocument;
import eu.europa.esig.dss.model.FileDocument;
import eu.europa.esig.dss.model.InMemoryDocument;
import eu.europa.esig.dss.model.ManifestEntry;
import eu.europa.esig.dss.model.ManifestFile;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.exception.IllegalInputException;
import eu.europa.esig.dss.utils.Utils;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ASiCUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ASiCUtils.class);
    public static final String MANIFEST_FILENAME = "Manifest";
    public static final String ASIC_MANIFEST_FILENAME = "ASiCManifest";
    public static final String ASIC_ARCHIVE_MANIFEST_FILENAME = "ASiCArchiveManifest";
    public static final String ASIC_EVIDENCE_RECORD_MANIFEST_FILENAME = "ASiCEvidenceRecordManifest";
    public static final String ASIC_XAdES_MANIFEST_FILENAME = "manifest";
    public static final String MIME_TYPE = "mimetype";
    public static final String MIME_TYPE_COMMENT = "mimetype=";
    public static final String META_INF_FOLDER = "META-INF/";
    public static final String PACKAGE_ZIP = "package.zip";
    public static final String SIGNATURE_FILENAME = "signature";
    public static final String SIGNATURES_FILENAME = "signatures";
    public static final String TIMESTAMP_FILENAME = "timestamp";
    public static final String EVIDENCE_RECORD_FILENAME = "evidencerecord";
    public static final String CADES_SIGNATURE_EXTENSION = ".p7s";
    public static final String TST_EXTENSION = ".tst";
    public static final String ER_ASN1_EXTENSION = ".ers";
    public static final String XML_EXTENSION = ".xml";
    public static final String SIGNATURES_XML = "META-INF/signatures.xml";
    public static final String OPEN_DOCUMENT_SIGNATURES = "META-INF/documentsignatures.xml";
    public static final String ASICE_METAINF_MANIFEST = "META-INF/manifest.xml";
    public static final String ASICE_METAINF_XADES_SIGNATURE = "META-INF/signatures001.xml";
    public static final String ASICE_METAINF_CADES_SIGNATURE = "META-INF/signature001.p7s";
    public static final String ASICE_METAINF_CADES_TIMESTAMP = "META-INF/timestamp001.tst";
    public static final String ASICE_METAINF_CADES_EVIDENCE_RECORD_ASN1 = "META-INF/evidencerecord001.ers";
    public static final String ASICE_METAINF_CADES_EVIDENCE_RECORD_XML = "META-INF/evidencerecord001.xml";
    public static final String ASICE_METAINF_CADES_MANIFEST = "META-INF/ASiCManifest001.xml";
    public static final String ASICE_METAINF_CADES_ARCHIVE_MANIFEST = "META-INF/ASiCArchiveManifest001.xml";
    public static final String ASICE_METAINF_EVIDENCE_RECORD_MANIFEST = "META-INF/ASiCEvidenceRecordManifest001.xml";
    public static final String SIGNATURE_P7S = "META-INF/signature.p7s";
    public static final String TIMESTAMP_TST = "META-INF/timestamp.tst";
    public static final String EVIDENCE_RECORD_ERS = "META-INF/evidencerecord.ers";
    public static final String EVIDENCE_RECORD_XML = "META-INF/evidencerecord.xml";
    public static final byte[] ZIP_PREFIX = new byte[]{80, 75};
    private static final byte[] MAGIC_DIR = new byte[]{80, 75, 5, 6};
    private static final int MAX_TO_READ = 65537 + MAGIC_DIR.length;

    private ASiCUtils() {
    }

    public static boolean isSignature(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(SIGNATURE_FILENAME) && !entryName.contains(MANIFEST_FILENAME);
    }

    public static boolean isTimestamp(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(TIMESTAMP_FILENAME) && entryName.endsWith(TST_EXTENSION);
    }

    public static boolean isEvidenceRecord(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(EVIDENCE_RECORD_FILENAME) && (entryName.endsWith(XML_EXTENSION) || entryName.endsWith(ER_ASN1_EXTENSION));
    }

    public static boolean isXmlEvidenceRecord(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(EVIDENCE_RECORD_FILENAME) && entryName.endsWith(XML_EXTENSION);
    }

    public static boolean isAsn1EvidenceRecord(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(EVIDENCE_RECORD_FILENAME) && entryName.endsWith(ER_ASN1_EXTENSION);
    }

    public static String getMimeTypeString(ASiCParameters asicParameters) {
        MimeType mimeType = ASiCUtils.getMimeType(asicParameters);
        return mimeType.getMimeTypeString();
    }

    public static String getZipComment(ASiCParameters asicParameters) {
        if (asicParameters.isZipComment()) {
            return ASiCUtils.getZipComment(ASiCUtils.getMimeTypeString(asicParameters));
        }
        return "";
    }

    public static String getZipComment(MimeType mimeType) {
        return ASiCUtils.getZipComment(mimeType.getMimeTypeString());
    }

    public static String getZipComment(String mimeTypeString) {
        return MIME_TYPE_COMMENT + mimeTypeString;
    }

    public static boolean isASiCMimeType(MimeType mimeType) {
        return MimeTypeEnum.ASICS.equals((Object)mimeType) || MimeTypeEnum.ASICE.equals((Object)mimeType);
    }

    public static boolean isOpenDocumentMimeType(MimeType mimeType) {
        return MimeTypeEnum.ODT.equals((Object)mimeType) || MimeTypeEnum.ODS.equals((Object)mimeType) || MimeTypeEnum.ODG.equals((Object)mimeType) || MimeTypeEnum.ODP.equals((Object)mimeType);
    }

    public static ASiCContainerType getASiCContainerType(MimeType asicMimeType) {
        Objects.requireNonNull(asicMimeType, "MimeType cannot be null!");
        if (MimeTypeEnum.ASICS.equals((Object)asicMimeType)) {
            return ASiCContainerType.ASiC_S;
        }
        if (MimeTypeEnum.ASICE.equals((Object)asicMimeType) || ASiCUtils.isOpenDocumentMimeType(asicMimeType)) {
            return ASiCContainerType.ASiC_E;
        }
        throw new IllegalArgumentException("Not allowed mimetype '" + asicMimeType.getMimeTypeString() + "'");
    }

    public static boolean isASiCE(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_E.equals((Object)asicParameters.getContainerType());
    }

    public static boolean isASiCS(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_S.equals((Object)asicParameters.getContainerType());
    }

    public static MimeType getMimeType(ASiCParameters asicParameters) {
        if (Utils.isStringNotBlank((String)asicParameters.getMimeType())) {
            return MimeType.fromMimeTypeString((String)asicParameters.getMimeType());
        }
        return ASiCUtils.isASiCE(asicParameters) ? MimeTypeEnum.ASICE : MimeTypeEnum.ASICS;
    }

    public static boolean filesContainCorrectSignatureFileWithExtension(List<String> filenames, String extension) {
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename) || !filename.endsWith(extension)) continue;
            return true;
        }
        return false;
    }

    public static boolean filesContainSignatures(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean filesContainTimestamps(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isTimestamp(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean filesContainEvidenceRecords(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isEvidenceRecord(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isAsicFileContent(List<String> filenames) {
        return ASiCUtils.filesContainCorrectSignatureFileWithExtension(filenames, CADES_SIGNATURE_EXTENSION) || ASiCUtils.filesContainCorrectSignatureFileWithExtension(filenames, XML_EXTENSION) || ASiCUtils.filesContainTimestamps(filenames);
    }

    public static boolean isZip(DSSDocument document) {
        boolean bl;
        block10: {
            if (document == null) {
                return false;
            }
            if (document instanceof DigestDocument) {
                return false;
            }
            InputStream is = document.openStream();
            try {
                bl = ASiCUtils.isZip(is);
                if (is == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new DSSException(String.format("Unable to determine whether the document with name '%s' represents a ZIP container. Reason : %s", document.getName(), e.getMessage()), (Throwable)e);
                }
            }
            is.close();
        }
        return bl;
    }

    public static boolean isZip(InputStream is) {
        Objects.requireNonNull(is, "InputStream cannot be null!");
        try {
            return Utils.startsWith((InputStream)is, (byte[])ZIP_PREFIX);
        }
        catch (IOException e) {
            throw new IllegalInputException("Unable to read the 2 first bytes", (Throwable)e);
        }
    }

    public static boolean isASiCWithXAdES(List<String> filenames) {
        return ASiCUtils.filesContainCorrectSignatureFileWithExtension(filenames, XML_EXTENSION);
    }

    public static boolean isASiCWithCAdES(List<String> filenames) {
        return ASiCUtils.filesContainCorrectSignatureFileWithExtension(filenames, CADES_SIGNATURE_EXTENSION) || ASiCUtils.filesContainTimestamps(filenames);
    }

    public static boolean isXAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(XML_EXTENSION);
    }

    public static boolean isCAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(CADES_SIGNATURE_EXTENSION);
    }

    public static boolean isContainerOpenDocument(DSSDocument archiveContainer) {
        DSSDocument mimetype = ASiCUtils.getMimetypeDocument(archiveContainer);
        return mimetype != null && ASiCUtils.isOpenDocument(mimetype);
    }

    private static DSSDocument getMimetypeDocument(DSSDocument archiveDocument) {
        List<DSSDocument> documents = ZipUtils.getInstance().extractContainerContent(archiveDocument);
        for (DSSDocument document : documents) {
            if (!ASiCUtils.isMimetype(document.getName())) continue;
            return document;
        }
        return null;
    }

    public static boolean isOpenDocument(DSSDocument mimeTypeDoc) {
        MimeType mimeType = ASiCUtils.getMimeType(mimeTypeDoc);
        if (mimeTypeDoc != null) {
            return ASiCUtils.isOpenDocumentMimeType(mimeType);
        }
        return false;
    }

    public static boolean areFilesContainMimetype(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isMimetype(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isMimetype(String entryName) {
        return MIME_TYPE.equals(entryName);
    }

    public static MimeType getMimeType(DSSDocument mimeTypeDocument) {
        MimeType mimeType;
        block9: {
            if (mimeTypeDocument == null) {
                return null;
            }
            InputStream is = mimeTypeDocument.openStream();
            try {
                byte[] byteArray = Utils.toByteArray((InputStream)is);
                String mimeTypeString = new String(byteArray, StandardCharsets.UTF_8);
                mimeType = MimeType.fromMimeTypeString((String)mimeTypeString);
                if (is == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new IllegalInputException(String.format("Unable to read mimetype file. Reason : %s", e.getMessage()), (Throwable)e);
                }
            }
            is.close();
        }
        return mimeType;
    }

    public static boolean isASiCSContainer(DSSDocument container) {
        return ASiCContainerType.ASiC_S.equals((Object)ASiCUtils.getContainerType(container));
    }

    public static boolean isASiCEContainer(DSSDocument container) {
        return ASiCContainerType.ASiC_E.equals((Object)ASiCUtils.getContainerType(container));
    }

    public static ASiCContainerType getContainerType(DSSDocument archiveContainer) {
        Objects.requireNonNull(archiveContainer, "Archive container shall be provided!");
        List<String> entryNames = ZipUtils.getInstance().extractEntryNames(archiveContainer);
        DSSDocument mimetypeDocument = null;
        if (ASiCUtils.areFilesContainMimetype(entryNames)) {
            mimetypeDocument = ASiCUtils.getMimetypeDocument(archiveContainer);
        }
        String zipComment = ASiCUtils.getZipComment(archiveContainer);
        int signedDocumentsNumber = ASiCUtils.getNumberOfSignedRootDocuments(entryNames);
        return ASiCUtils.getContainerType(archiveContainer.getMimeType(), mimetypeDocument, zipComment, signedDocumentsNumber);
    }

    public static boolean isASiCSContainer(ASiCContent asicContent) {
        return ASiCContainerType.ASiC_S.equals((Object)ASiCUtils.getContainerType(asicContent));
    }

    public static boolean isASiCEContainer(ASiCContent asicContent) {
        return ASiCContainerType.ASiC_E.equals((Object)ASiCUtils.getContainerType(asicContent));
    }

    public static ASiCContainerType getContainerType(ASiCContent asicContent) {
        Objects.requireNonNull(asicContent, "ASiCContent shall be provided!");
        if (asicContent.getContainerType() != null) {
            return asicContent.getContainerType();
        }
        MimeType containerMimeType = asicContent.getAsicContainer() != null ? asicContent.getAsicContainer().getMimeType() : null;
        return ASiCUtils.getContainerType(containerMimeType, asicContent.getMimeTypeDocument(), asicContent.getZipComment(), Utils.collectionSize(asicContent.getRootLevelSignedDocuments()));
    }

    private static int getNumberOfSignedRootDocuments(List<String> containerEntryNames) {
        int signedDocumentCounter = 0;
        for (String documentName : containerEntryNames) {
            if (documentName.contains("/") || ASiCUtils.isMimetype(documentName)) continue;
            ++signedDocumentCounter;
        }
        return signedDocumentCounter;
    }

    private static ASiCContainerType getContainerType(MimeType containerMimeType, DSSDocument mimetypeDocument, String zipComment, int rootSignedDocumentsNumber) {
        ASiCContainerType containerType = ASiCUtils.getContainerTypeFromMimeTypeDocument(mimetypeDocument);
        if (containerType != null) {
            return containerType;
        }
        containerType = ASiCUtils.getContainerTypeFromZipComment(zipComment);
        if (containerType != null) {
            return containerType;
        }
        if (rootSignedDocumentsNumber > 1) {
            return ASiCContainerType.ASiC_E;
        }
        containerType = ASiCUtils.getContainerTypeFromMimeType(containerMimeType);
        if (containerType != null) {
            return containerType;
        }
        LOG.info("Unable to define the ASiC Container type with its properties. Assume type based on root-level documents...");
        if (rootSignedDocumentsNumber == 1) {
            containerType = ASiCContainerType.ASiC_S;
        } else {
            LOG.warn("The provided container does not contain signer documents on the root level!");
        }
        return containerType;
    }

    private static ASiCContainerType getContainerTypeFromZipComment(String zipComment) {
        int indexOf;
        if (Utils.isStringNotBlank((String)zipComment) && (indexOf = zipComment.indexOf(MIME_TYPE_COMMENT)) > -1) {
            String asicCommentMimeTypeString = zipComment.substring(MIME_TYPE_COMMENT.length() + indexOf);
            MimeType mimeTypeFromZipComment = MimeType.fromMimeTypeString((String)asicCommentMimeTypeString);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromZipComment);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeTypeDocument(DSSDocument mimetype) {
        if (mimetype != null) {
            MimeType mimeTypeFromEmbeddedFile = ASiCUtils.getMimeType(mimetype);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromEmbeddedFile);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeType(MimeType mimeType) {
        if (ASiCUtils.isASiCMimeType(mimeType)) {
            return ASiCUtils.getASiCContainerType(mimeType);
        }
        return null;
    }

    public static boolean isManifest(String fileName) {
        return fileName.startsWith(META_INF_FOLDER) && fileName.contains(ASIC_MANIFEST_FILENAME) && fileName.endsWith(XML_EXTENSION);
    }

    public static boolean isArchiveManifest(String fileName) {
        return fileName.startsWith(META_INF_FOLDER) && fileName.contains(ASIC_ARCHIVE_MANIFEST_FILENAME) && fileName.endsWith(XML_EXTENSION);
    }

    public static boolean isEvidenceRecordManifest(String fileName) {
        return fileName.startsWith(META_INF_FOLDER) && fileName.contains(ASIC_EVIDENCE_RECORD_MANIFEST_FILENAME) && fileName.endsWith(XML_EXTENSION);
    }

    public static boolean coversSignature(ManifestFile manifestFile) {
        for (ManifestEntry manifestEntry : manifestFile.getEntries()) {
            if (!ASiCUtils.isSignature(manifestEntry.getUri())) continue;
            return true;
        }
        return false;
    }

    public static void addOrReplaceDocument(List<DSSDocument> documentList, DSSDocument newDocument) {
        for (int i = 0; i < documentList.size(); ++i) {
            if (!newDocument.getName().equals(documentList.get(i).getName())) continue;
            documentList.set(i, newDocument);
            return;
        }
        documentList.add(newDocument);
    }

    public static ASiCContent ensureMimeTypeAndZipComment(ASiCContent asicContent, ASiCParameters asicParameters) {
        if (asicContent.getMimeTypeDocument() == null) {
            MimeType mimeType = ASiCUtils.getMimeType(asicContent, asicParameters);
            DSSDocument mimetypeDocument = ASiCUtils.createMimetypeDocument(mimeType);
            asicContent.setMimeTypeDocument(mimetypeDocument);
        }
        if (Utils.isStringEmpty((String)asicContent.getZipComment())) {
            String zipComment = ASiCUtils.getZipComment(asicContent, asicParameters);
            asicContent.setZipComment(zipComment);
        }
        return asicContent;
    }

    private static MimeType getMimeType(ASiCContent asicContent, ASiCParameters asicParameters) {
        MimeType mimeType = null;
        DSSDocument mimeTypeDocument = asicContent.getMimeTypeDocument();
        if (mimeTypeDocument != null) {
            mimeType = ASiCUtils.getMimeType(mimeTypeDocument);
        }
        if (mimeType == null) {
            Objects.requireNonNull(asicParameters, "ASiCParameters shall be present for the requested operation!");
            mimeType = ASiCUtils.getMimeType(asicParameters);
        }
        return mimeType;
    }

    private static String getZipComment(ASiCContent asicContent, ASiCParameters asicParameters) {
        String zipComment = asicContent.getZipComment();
        if (Utils.isStringNotEmpty((String)zipComment)) {
            return zipComment;
        }
        if (asicParameters != null) {
            return ASiCUtils.getZipComment(asicParameters);
        }
        return "";
    }

    private static DSSDocument createMimetypeDocument(MimeType mimeType) {
        byte[] mimeTypeBytes = mimeType.getMimeTypeString().getBytes(StandardCharsets.UTF_8);
        InMemoryDocument mimetypeDocument = new InMemoryDocument(mimeTypeBytes, MIME_TYPE);
        ContainerEntryDocument zipEntryDocument = new ContainerEntryDocument((DSSDocument)mimetypeDocument);
        zipEntryDocument.getZipEntry().setCompressionMethod(0);
        return zipEntryDocument;
    }

    public static List<DSSDocument> getRootLevelSignedDocuments(ASiCContent asicContent) {
        List<DSSDocument> signedDocuments = asicContent.getSignedDocuments();
        if (Utils.isCollectionEmpty(signedDocuments)) {
            return Collections.emptyList();
        }
        if (Utils.collectionSize(signedDocuments) == 1) {
            return signedDocuments;
        }
        return ASiCUtils.getRootLevelDocuments(signedDocuments);
    }

    public static List<DSSDocument> getRootLevelDocuments(List<DSSDocument> documents) {
        ArrayList<DSSDocument> rootDocuments = new ArrayList<DSSDocument>();
        for (DSSDocument document : documents) {
            String documentName = document.getName();
            if (documentName == null || documentName.contains("/") || MIME_TYPE.equals(documentName)) continue;
            rootDocuments.add(document);
        }
        return rootDocuments;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static String getZipComment(DSSDocument archiveContainer) {
        long fileLength = ASiCUtils.getFileLength(archiveContainer);
        try (InputStream is = archiveContainer.openStream();){
            long toSkip;
            long skipped;
            if (fileLength > (long)MAX_TO_READ && (skipped = is.skip(toSkip = fileLength - (long)MAX_TO_READ)) != toSkip) {
                throw new IOException("Different amount of bytes have been skipped!");
            }
            byte[] buffer = DSSUtils.toByteArray((InputStream)is);
            if (Utils.isArrayEmpty((byte[])buffer)) {
                LOG.warn("An empty container obtained! Unable to extract zip comment.");
                String string = null;
                return string;
            }
            int len = buffer.length;
            for (int ii = len - 22; ii >= 0; --ii) {
                boolean isMagicStart = true;
                for (int jj = 0; jj < MAGIC_DIR.length; ++jj) {
                    if (buffer[ii + jj] == MAGIC_DIR[jj]) continue;
                    isMagicStart = false;
                    break;
                }
                if (!isMagicStart) continue;
                int commentLen = buffer[ii + 20] + buffer[ii + 21] * 256;
                int realLen = len - ii - 22;
                if (commentLen != realLen) {
                    LOG.warn("WARNING! ZIP comment size mismatch: directory says len is {}, but file ends after {} bytes!", (Object)commentLen, (Object)realLen);
                }
                if (realLen == 0) {
                    String string = null;
                    return string;
                }
                String string = new String(buffer, ii + 22, realLen);
                return string;
            }
        }
        catch (IOException e) {
            throw new DSSException(String.format("Unable to read content of document with name '%s'. Reason : %s", archiveContainer.getName(), e.getMessage()));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Zip comment is not found in the provided container with name '{}'", (Object)archiveContainer.getName());
        }
        return null;
    }

    private static long getFileLength(DSSDocument archiveContainer) {
        long l;
        block9: {
            if (archiveContainer instanceof FileDocument) {
                FileDocument doc = (FileDocument)archiveContainer;
                return doc.getFile().length();
            }
            InputStream is = archiveContainer.openStream();
            try {
                l = Utils.getInputStreamSize((InputStream)is);
                if (is == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new DSSException("Unable to compute archive size", (Throwable)e);
                }
            }
            is.close();
        }
        return l;
    }

    public static List<ManifestEntry> toSimpleManifestEntries(List<DSSDocument> documents) {
        ArrayList<ManifestEntry> entries = new ArrayList<ManifestEntry>();
        for (DSSDocument document : documents) {
            Objects.requireNonNull(document, "DSSDocument cannot be null!");
            ManifestEntry entry = new ManifestEntry();
            entry.setUri(document.getName());
            entry.setMimeType(document.getMimeType());
            entry.setFound(true);
            entry.setDocumentName(document.getName());
            entries.add(entry);
        }
        return entries;
    }
}

